;;;*****************************************************************************
;;;
;;; Dplacement d'un point en deux dimensions, bas sur le principe des
;;; picycles de Ptolme
;;;
;;; Version temporise en secondes
;;;
;;; Denis Lorrain, SONVS / CNSM Lyon
;;; Rvision LispWorks CAPI, HfM Karlsruhe, 2009
;;; Ralis en LispWorks par recours  mp::process-wait-with-timeout
;;;
;;;*****************************************************************************

;;   "Platon [...], en imposant aux mouvements des corps clestes l'obligation
;; d'tre circulaires, uniformes et rgls, a propos aux mathmaticiens le
;; problme suivant : quelles sont les hypothses qui, par des mouvements
;; uniformes, circulaires et rgls, pourront sauver les faits observs pour les
;; plantes ?" [Simplicius : "Commentaire au second livre du Trait du ciel
;; d'Aristote", in Verdet, J.-P. (d.), 1993, <Astronomie et astrophysique>,
;; Larousse : 158].
;;   Par la suite, d'Eudoxe  Aristote et ses disciples, on labora un systme
;; complexe de sphres thres mobiles homocentriques. "Aristote, en effet, qui
;; considrait que tous les corps clestes doivent tourner autour du milieu du
;; tout, tait fervent partisan de l'hypothse des sphres tournantes, supposes
;; homocentriques au tout [...]" [<ibid.> : 159]. De quarante-sept 
;; cinquante-cinq sphres se meuvent les unes  l'intrieur des autres  partir
;; de la sphre extrieure portant les toiles fixes. Regroupes par ensembles
;; de trois  cinq, ces sphres tentent de rendre compte des mouvements de
;; chaque astre (soleil, lune, plantes) : dans chaque ensemble, une seule
;; sphre porte l'astre proprement dit, les mouvements des autres se combinant
;; pour gouverner les mouvements de celle-ci.
;;   Malheureusement, dans ces sphropes, "les faits observs ne sont nullement
;; sauvs, non seulement ceux qui n'ont t aperus que plus tard, mais mme pas
;; ceux qui taient connus depuis longtemps et parfaitement admis" [<ibid.> :
;; 168]. "La condamnation de l'hypothse des sphres tournantes par les
;; gnrations postrieures repose sur le fait qu'elle ne sauvait ni la
;; variation en profondeur ni l'irrgularit des mouvements ; on abandonna donc
;; les sphres homocentriques tournantes et l'on supposa des excentriques et des
;; picycles (il se peut que l'hypothse des excentriques ait dj t imagine
;; par les Pythagoriciens [...])" [<ibid.> : 171].
;;   "Chacune [des] deux hypothses [des excentriques et picycles], prise
;; sparment, satisfait  l'objectif de l'astronomie, sauf que, pour la lune,
;; il est besoin d'utiliser les deux  la fois [...]. Ces hypothses ont en
;; outre le mrite d'tre plus simples que les [sphropes] ; sans exiger
;; l'introduction de tant de corps clestes, elles sauvent fort bien les faits
;; observs [...]. En revanche elles ne respectent pas l'axiome d'Aristote qui
;; veut que tout corps anim d'un mouvement circulaire tourne autour du milieu
;; du tout" [<ibid.> : 173].

;;; Principe

;;   Un premier cercle, dfrent, est en rotation autour de son centre. Un point
;; <P1> de la circonfrence du dfrent est  son tour le centre d'un second
;; cercle, ou picycle, lui-mme en rotation autour de son propre centre mobile
;; <P1>. L'picycle peut  son tour tre dfrent : un point <P2> de sa
;; circonfrence tant au centre d'un troisime cercle en rotation autour de
;; <P2> etc.
;;   Comme le dit Copernic, postulant des orbites circulaires : "l'orbe de la
;; lune [...] est comme [l']picycle" de l'orbite de la terre [<ibid.> : 222].

;;------------------------------------------------------------------------------
;; Cercle trigonimtrique modifi
;;------------------------------------------------------------------------------
;;         360=0
;;
;;    315          45
;;
;;  270      *       90
;;
;;    225          135
;;
;;           180
;; N.B. : modifications par rapport au cercle trigonomtrique normal :
;;    - angles exprims en degrs plutt qu'en radians ;
;;    - 0 := "12:00" ;
;;    - vitesse de rotation positive := dans le sens des aiguilles d'une montre.
;; Les consquences de ces modifications sont prises en compte dans la fonction
;; epicycles ci-dessous.

(defvar deupi (* 2 pi))

;;   Conversion de coordonnes polaires vers cartsiennes. Rend une paire
;; pointe (x . y).
(defun pol->cart (rayon degres)
  (let ((radians (* (/ degres 360.0) deupi)))
    (cons (* rayon (cos radians)) (* rayon (sin radians)))))

;;   Calcul du dplacement d'un point rsultant des rotations d'un nombre <n>
;; quelconque de cercles : un dfrent et <n>-1 picycles. Le point considr
;; est <P<n>>, sur la circonfrence du dernier picycle.
;;   Appel :
;;    (epicycles <rayons> <secspartour> <angles> <dure> [<echparsec>]), o
;;        <rayons>      := liste des rayons des <n> cercles.
;;        <secspartour> := liste des vitesses de rotation des <n> cercles, en
;;                         secondes par tour (360), en sens positif ou ngatif.
;;        <angles>      := liste des positions angulaires initiales des points
;;                         <P<m>>. Le point <P<m>> du <m>-ime cercle est le
;;                         centre de son picycle : du (<m>+1)-ime cercle.
;;        <dure>       := dure pendant laquelle on calculera/chantillonnera
;;                         les rvolutions des <n> cercles, donc la position de
;;                         <P<n>>. En secondes **relles**.;<- variante CLCE-IV
;;        [<echparsec>] := nombre d'chantillons discrets par seconde [10 par
;;                         dfaut].
;;   La fonction rend un chantillonnage de la position du point <P<n>> du
;; <n>-ime cercle - le dernier. Les positions sont rendues sous forme brute :
;; en coordonnes cartsiennes par rapport  une origine [0;0] situe au centre
;; du premier cercle dfrent. Les coordonnes sont rendues dans une liste de
;; deux sous-listes : la premire contenant les coordonnes successives en x, la
;; seconde en y.
;;   Exemples pour tests :
;;    (epicycles '(1) '(1) '(0) 1.5 8)
;;    (epicycles '(1 0.1 0.05) '(1 1/4 -1/4) '(0 90 -270) 1.5 8)
;;    (epicycles '(2 .5 .4 .3) '(1 -1/6 1/6 1/12) '(0 -270 -180 270) 5 8)
(defun epicycles (rayons secspartour angles duree
                 &optional (echparsec 10)
                 &aux ncercles vecray vecincr vecangl rezX rezY )
  (setq ncercles (list-length rayons))
  (setq  vecray (make-array ncercles :initial-contents rayons))
  (setq rezX (list (/ echparsec -360.0)))  ;-360 pour inverser signe incrments
  (setq vecincr (make-array ncercles :initial-contents
                            (mapcar #'/ (mapcar #'* secspartour
                                                    (nconc rezX rezX)))))
  (setq vecangl (make-array ncercles
                 :initial-contents                       ;conversion tous angles
                 (mapcar #'(lambda (angle)               ;de dpart selon...
                             (mod (+ 90.0 (mod (- angle) 360.0)) 360.0))
                         angles)))                       ;... 0 := "12:00"
  (setq rezX nil)
  (dotimes (n (1+ (round (* duree echparsec)))
              (list (reverse rezX) (reverse rezY)) )
    (do ((n 0 (1+ n))
         (cumX 0)
         (cumY 0)
         (coord nil)
         (langle 0)  )
        ((= n ncercles) (progn (push cumX rezX) (push cumY rezY)))
      (setq langle (aref vecangl n))
      (setq coord (pol->cart (aref vecray n) langle))
      (incf cumX (car coord))
      (incf cumY (cdr coord))
      (setf (aref vecangl n) (+ langle (aref vecincr n))))))
      
;;   Pour tests : imprime des colonnes de coordonnes en x et y.
;; Exemples pour tests :
;;    (imprcoord (epicycles '(1 0.1 0.05) '(1 1/4 -1/4) '(0 90 -270) 1.5 8))
;;    (imprcoord (epicycles '(2 .5 .4 .3) '(1 -1/6 1/6 1/12) '(0 -270 -180 270) 5 8))
(defun imprcoord (lili &aux zix zyg)
  (setq zix (car lili))
  (setq zyg (cadr lili))
  (format t "~8T===X===~18T===Y===~%")
  (dotimes (n (list-length zix))
  (format t "~3D : ~9,3F ~9,3F~%" n (pop zix) (pop zyg))))

;;; Transformation de la sortie de la fonction epicycles : pour compatibilit
;;; avec un mode de spatialisation CSound

;;------------------------------------------------------------------------------
;; Ttraphonie mobile cartsienne CSound
;;------------------------------------------------------------------------------
;;         y
;;
;;         
;; [0;1]:=A          B:=[1;1]
;;         
;;         
;;         
;;         
;; [0;0]:=D__________C:=[0;1]__ x

;;   Normalise  [-1/2;1/2] des sous listes de nombres quelconques, chacune
;; indpendamment, puis les translate en [0;1].
(defun trnormalili (lili)
  (let* ((lidemi (list 0.5))
         (trnslt (nconc lidemi lidemi)))
    (labels ((interne (lil)
               (if (endp lil) ()
               (let* ((lali (car lil))
                     (fact (list (* 2 (apply #'max (mapcar #'abs lali))))))
                 (cons (mapcar #'+ (mapcar #'/ lali (nconc fact fact))
                                   trnslt)
                       (interne (cdr lil)))))))
      (interne lili))))

;;; Interface graphique de dmonstration

;;   Dfinitions d'une variable globale pour la fentre qui sera utilise par la
;; fonction dessin ci-dessous.

(defparameter *fen* nil)

;;   Pour transformer les deux sous-listes de <lili> de coordonnes cartsiennes
;; en norme Csound (dans [0;1]), vers affichage dans [0;<dim>] avec origine en
;; haut  gauche, dcale.
(defun coord->graf (lili dim)
  (let* ((tp1 (list 1.0))
         (cc1 (nconc tp1 tp1))
         (tp2 (list dim))
         (cc2 (nconc tp2 tp2))
         (tp3 (list 10))
         (cc3 (nconc tp3 tp3)) )
    (list (mapcar #'+ cc3 (mapcar #'round
                                  (mapcar #'* cc2 (car lili))))
          (mapcar #'+ cc3 (mapcar #'round
                                  (mapcar #'* cc2
                                          (mapcar #'- cc1 (cadr lili))))))))

;;   Pour tracer les points dans la fentre
(defun dessin (lili echparsec &aux sec)
  (setq *fen* (capi::contain
               (make-instance 'capi::output-pane)
               :best-width 683                               ;683 => <dim> = 650
               :best-height 683
               :title (string (gensym "PTOLEMAEVS-"))))
  (setq sec (/ 1.0 echparsec))
  (let ((zixs (car lili))
        (zygs (cadr lili)))
    (dotimes (n (list-length zixs) 'ok)
      (capi::apply-in-pane-process *fen*
                                   'gp::draw-string *fen*
                                   (format nil "*")
                                   (pop zixs) (pop zygs))
      (mp::process-wait-with-timeout "after" sec)        ;<<<=== pour temposiser
      )))

;;; Tests...
;;;    (epicycles <rayons> <secspartour> <angles> <dure> [<echparsec>])     

;(dessin (coord->graf (trnormalili (epicycles '(1) '(1) '(0) 1.5 8)) 650))
;(dessin (coord->graf (trnormalili (epicycles '(1) '(3) '(0) 3.1 50)) 650))
;(dessin (coord->graf (trnormalili (epicycles '(1) '(1.05) '(0) 10.5 8)) 650))
;(dessin (coord->graf (trnormalili
;   (epicycles '(1 0.3) '(2.05 1.76) '(0 90) 10.5)) 650))
;(dessin (coord->graf (trnormalili
;   (epicycles '(1 0.3) '(4.05 0.9) '(0 90) 10.5 50)) 650))
;(dessin (coord->graf (trnormalili
;   (epicycles '(1 1.3) '(3.35 1.9) '(10 90) 10.5 50)) 650))
;(dessin (coord->graf (trnormalili ;type spirale vari
;   (epicycles '(1 1.3 0.76) '(3.35 1.9 1.55) '(10 90 -27) 10.5 50)) 650))
;(dessin (coord->graf (trnormalili
;   (epicycles '(1 0.33 0.15) '(-2 0.4 -.33) '(44 266 180) 6.1 100)) 650))
;(dessin (coord->graf (trnormalili
;   (epicycles '(1 0.33 0.15) '(-2 0.4 -.13) '(44 266 180) 6.1 100)) 650))

;;   Fonction interface finale...
(defun Ptolemee (rayons secspartour angles duree &optional (echparsec 10))
  (dessin
   (coord->graf (trnormalili
                 (epicycles rayons secspartour angles duree echparsec)) 650)
   echparsec))

;(ptolemee '(1) '(-2) '(225) 2)
;(ptolemee '(1) '(-2) '(225) 2 20)
;(ptolemee '(1 0.3) '(2.05 1.76) '(0 90) 50.5 23)
;(ptolemee '(1 0.3) '(42.05 0.56) '(0 90) 22.05 40) ;type slinky...
;(ptolemee '(1 0.3) '(2.05 -1.76) '(0 90) 50.5 25) ;type plantaire
;(ptolemee '(1385/6 1181/60 394/60) '(5 -5 2.5) '(0 0 0) 5 40) ;Saturne Copernic [1]
;(ptolemee '(25 1181/60 394/60) '(5 -5 2.5) '(0 0 0) 5 40) ;8
;(ptolemee '(25 1181/60 394/60) '(5 -5 2.5) '(40 40 40) 5 40) ;8
;(ptolemee '(25 1181/60 394/60) '(5 -5 2.5) '(136 140 133) 5 40) ;8
;(ptolemee '(25 1181/60 394/60) '(5 -5 2.5) '(136 147 153) 5 40) ;8
;(ptolemee '(25 1181/60 394/60) '(5 -5 2.5) '(126 147 173) 5 40) ;... goutte
;(ptolemee '(25 1181/60 394/60) '(5 -5 2.5) '(110 110 40) 5 40) ;... goutte
;(ptolemee '(25 1181/60 394/60) '(5 -5 2.5) '(0 90 0) 5 40) ;... banana
;(ptolemee '(25 1181/60 394/60) '(5 -5 2.5) '(0 90 33) 5 40) ;... banana
;(ptolemee '(25 1181/60 394/60) '(5 -5 2.5) '(0 47 113) 5 40) ;... goutte
;(ptolemee '(1565/12 606/60 202/60) '(5 -5 2.5) '(0 0 0) 5 40) ;Jupiter Copernic [1]
;(ptolemee '(25 606/60 202/60) '(5 -5 2.5) '(0 0 0) 5 40) ;ovode
;(ptolemee '(25 606/60 202/60) '(5 -5 2.5) '(0 90 0) 5 40) ;ovode
;(ptolemee '(25 606/60 202/60) '(5 -5 2.5) '(90 90 170) 5 40) ;ovode
;(ptolemee '(25 606/60 202/60) '(-5 5 -2.5) '(270 217 350) 5 40) ;ovode
;(ptolemee '(38 334/60 111/60) '(5 -5 2.5) '(0 0 0) 5 40) ;Mars Copernic [1]
;(ptolemee '(25 334/60 111/60) '(5 -5 2.5) '(0 0 0) 5 40)
;(ptolemee '(25 334/60 111/60) '(5 -5 2.5) '(90 33 250) 5 40)
;(ptolemee '(18 3/4 1/4) '(5 5 10/3) '(0 0 0) 10 40) ;Venus Copernic [1]
;(ptolemee '(18 3/4 1/4) '(5 5 10/3) '(0 60 110) 10 40)
;(ptolemee '(47/5 101/60 34/60) '(5 5 1825/176) '(0 0 0) 20 40) ;Mercure Copernic [1]
;(ptolemee '(47/5 101/60 34/60) '(5 5 1825/176) '(0 33 152) 20 40)
;(ptolemee '(1 0.21) '(2.05 -1.36) '(0 90) 50 25) ;type cardiode -> toile
;(ptolemee '(1 0.21) '(2.05 -1.36) '(0 90) 25 50) ;la mme plus finement ch.
;(ptolemee '(1 0.2) '(2.05 -2.15) '(0 -90) 6.5 80) ;type plantaire
;(ptolemee '(1 0.2) '(2.05 -2.15) '(180 -90) 6.5 80) ;type plantaire
;(ptolemee '(1 0.2 0.05) '(2.05 -2.15 2.33) '(0 -90 187) 6.5 80) ;type plantaire
;(ptolemee '(1 0.2 0.35) '(2.05 -2.15 2.33) '(0 -90 187) 12 80) ;type spirale
;(ptolemee '(1 1.2 0.35) '(2.05 -2.15 2.33) '(0 -90 187) 6.5 80) ;type spirale
;(ptolemee '(1 1.3 0.76) '(3.35 1.9 1.55) '(10 90 -27) 10.5 50) ;type spirale
;(ptolemee '(1 0.33 0.15) '(-2 0.4 -.33) '(44 266 180) 6.1 200);loopings
;(ptolemee '(1 0.33 0.15) '(-2.33 0.4 -.13) '(44 266 180) 6.1 200);loopings
;(ptolemee '(1 0.33 0.15) '(-2.33 0.4 -.13) '(44 266 180) 5 400);idem + fin
;(ptolemee '(1 0.2) '(10 -0.47) '(44 266) 20.1 100);loop. simples rguliers : dmo
;(ptolemee '(1 0.2 0.05) '(10 -0.67 0.13) '(44 266 -21) 20 100);loop. doubles rguliers rosace : dmo
;(ptolemee '(0.2 0.05 1) '(10 -0.67 0.13) '(44 266 -21) 20 100);chane par sauts : dmo
;(ptolemee '(0.2 0.05 1) '(10 -0.67 0.13) '(44 266 -21) 13 150);idem + fin
;(ptolemee '(0.2 0.05 1) '(-0.67 10 0.13) '(44 266 -21) 10 200);idem ++ fin
;(ptolemee '(0.05 0.2 1) '(0.13 -0.67 10) '(44 266 -21) 20.1 100);loop. doubles rguliers rosace : dmo

;; [1] "Commentariolus", in Verdet, J.-P. (d.), 1993, <Astronomie et
;;     astrophysique>, Larousse : 197 sq.]. Interprtations...

;;; Alatoire...

;;   Cf. ped.12.

(defun faire-liste (fonction &optional (nbr 10))
  (labels ((interne () (cond
            ((= 0 nbr) ())
            (t (decf nbr)
               (cons (eval fonction) (interne))))))
    (interne)))

(defun inrect (i1 i2)
"
;    inrect <i1> <i2>, ou <i1> < <i2>, et tous deux sont des entiers
;        => un entier entre <i1> et (<i2>-1) : dans [<i1>,<i2>[.
"
  (+ i1 (random (- i2 i1))))

(defun PtolAlea (&optional (n 3))
  (let ((rayons (faire-liste '(random 1.0) n))
        (vitess (faire-liste '(inrect -5.0 5.0) n))
        (angles (faire-liste '(inrect 0.0 360.0) n)))
    (format t "(ptolemee '~S '~S '~S 10 50)~%" rayons vitess angles)
    (ptolemee rayons vitess angles 10 50)))

;; (PtolAlea [<n>]) : produit un appel  ptolemee avec <n> (pi)cycle(s) de
;; paramtres alatoires contrls. Ci-dessous : appels produits par PtolAlea.

;; (PtolAlea 1)
;un seul cycle (ptolemee '(0.2833560244801566) '(3.953601250338556) '(148.07691452666543) 10 50)
;; (PtolAlea 2)
;quasi cercles (ptolemee '(0.0082435235534674 0.5088153137058304) '(-0.9764534718849758 -4.093212744394713) '(291.56649994024355 37.155708814329856) 10 50)
;loopings serrs (ptolemee '(0.32035924740202587 0.5941984366517485) '(-2.0046616103586956 -4.803201537212733) '(206.3055252401261 316.57670430201426) 22 50)
;; (PtolAlea)
;rosace (ptolemee '(0.7861143663266942 0.7761889896843377 0.04716786192202543) '(-1.6506264110107467 -0.4990059984829385 -3.3236130109496655) '(313.1444128475048 173.41885248835916 120.43108038684554) 10 50)
;rosace (ptolemee '(0.10316966679537606 0.048717217610622233 0.027715729601151737) '(-2.887148350267016 -2.8430689272341674 2.025006668688741) '(4.563079931890741 345.5441748081598 169.90556167421036) 10 50)
;bretzel (ptolemee '(0.49641587677675464 0.16561231047960429 0.07240051230862832) '(2.013891631763226 -4.197959730770364 2.019292017480902) '(83.47144976904886 359.74395029199115 265.1505391023132) 10 50)
;atome (ptolemee '(0.10316966679537606 0.048717217610622233 0.027715729601151737) '(-2.887148350267016 -2.8430689272341674 2.025006668688741) '(4.563079931890741 345.5441748081598 169.90556167421036) 10 50)
;cursive (ptolemee '(0.7997347895051069 0.15575373574300433 0.9319434853865644) '(3.540152979158149 2.952334578991559 -2.9957324415365836) '(287.3729828137401 248.82184201632037 126.953829837877) 10 50)
;zagzig (ptolemee '(0.4739575376249059 0.788012684859359 0.6366799932656437) '(-2.8449336399229868 2.9489603972974434 2.489455542507498) '(131.36237764984813 112.08178749695753 289.0336832572402) 10 50)
;4 coins (ptolemee '(0.11802766494085577 0.4199930344149341 0.7841382376554203) '(2.2143430397181643 -1.4546937056904543 4.533406457925972) '(98.30417964291607 339.10538989883923 189.77611308875703) 10 50)
;Dollar des sables (ptolemee '(0.37967700023143924 0.6572906927341422 0.2496425309211808) '(3.584282116190124 4.149282982850975 -0.24897125595988978) '(147.40980958083608 90.43795930303254 188.14812352535895) 20 100)
;plantaire (ptolemee '(0.6543629107146764 0.5655526380344343 0.5871200740097464) '(-3.8227361791259353 3.3364494858988554 -3.5913842145193073) '(70.49341579991098 156.5714993263995 89.11037647930975) 10 50)
;papillon (ptolemee '(0.7283912314630882 0.600710646436942 0.8730473348368841) '(4.426193944041048 4.0843019431164524 -1.4254073359545454) '(151.32680839174765 25.381760048459768 120.74545204778286) 10 50)
;triangle strilet (ptolemee '(0.6135867249473682 0.8180131033398456 0.008714210258264486) '(-0.8616859575019884 1.721689676347542 4.780077058412278) '(310.6062806967763 55.75607265486582 268.18889451807155) 10 50)
;quasi cercles (ptolemee '(0.09166488068488404 1.189559700320586E-4 0.41679740402251) '(4.047169534577373 0.9454948007045214 0.8615840521856839) '(260.2247950066343 93.28077229350949 184.59807370714486) 10 50)
;; (PtolAlea 4)
;toile aller-retours (ptolemee '(0.984273681387815 0.5561632710623802 0.502939245669058 0.058914000278178606) '(4.978028237778746 -2.4056963946776397 3.0456831828610653 4.848129246101024) '(63.54903180813236 307.70390389059463 162.65665777522446 200.56687399850577) 20 50)
;ionique (ptolemee '(0.5775148562668168 0.45258873040338743 0.5492375093498034 0.5432234677359898) '(3.7425793683949475 -3.2190859470871946 1.1628864634090785 -4.159911906644538) '(243.99021797625386 55.50458337274302 22.06410017249007 273.4295587864324) 10 50)
;X-patineuse (ptolemee '(0.7013891631763226 0.0802040269229636 0.7019292017480903 0.23186513824735794) '(4.992887508110865 2.3652927528420333 -3.968303332046239 -4.512827823893778) '(9.977662656414626 76.06265939038742 77.64951861956997 252.9002400727947) 10 50)
;boucles (ptolemee '(0.012675222033029836 0.9598449300226661 0.4719598935394732 0.4449511998238996) '(1.8493507557536493 -4.016143826797872 4.785476624566087 0.7932936321819088) '(207.9560708402769 100.31376842467816 111.67935824334502 301.44222054899154) 10 50)
;oeuf (ptolemee '(0.7997347895051069 0.15575373574300433 0.9319434853865644 0.854015297915815) '(2.952334578991559 -2.9957324415365836 2.982582855937226 1.9117178337866765) '(126.953829837877 170.6247135449661 283.68456654936926 229.20479757563174) 10 50)
;cursive (ptolemee '(0.2155066360077013 0.7948960397297443 0.7489455542507498 0.36489549347180034) '(-1.8866170139734018 3.0287134238122277 -3.8197233505914423 -0.8000696558506597) '(282.2897655559513 259.7163494298539 127.63102659514365 343.202632485335) 10 50)
;mouche (ptolemee '(0.27306716567476685 0.9419594163856645 0.5271558696909917 0.37967700023143924) '(1.572906927341422 -2.503574690788192 3.584282116190124 4.149282982850975) '(171.03703478544398 147.40980958083608 90.43795930303254 188.14812352535895) 10 50)
;femme (ptolemee '(0.6543629107146764 0.5655526380344343 0.5871200740097464 0.11772638208740649) '(3.3364494858988554 -3.5913842145193073 -3.041849561113584 -0.6507916853777909) '(89.11037647930975 262.22084332671176 216.25583271729914 314.2970405412783) 10 50)
;; (PtolAlea 5)
;foetus (ptolemee '(0.31984449982614166 0.9392337779313955 0.47001794813204173 0.9624179153934455 0.7425930532488842) '(1.0169779435116393 -3.1067422928414166 0.7676663729691242 2.011633304900738 2.0288864467831083) '(179.39813006557176 261.92367870718925 107.69485158416416 33.74525054124231 343.50326721277696) 10 50)
;palmier (ptolemee '(0.5401992304999367 0.5005345279360833 0.26030399512556196 0.4817574163559031 0.06950835151506281) '(-1.3634711436246914 -0.6835741469330605 -2.0811091122796928 3.9384345876768876 -1.1566539037165304) '(49.880204964206 276.43097079625534 284.06863272947936 111.9051935372971 10.508882869982873) 10 50);; (PtolAlea 6)
;carrs (ptolemee '(0.6517475818316407 0.9711035787520996 0.2555692889831388 0.555320956847297 0.34298749704470394) '(2.2421316564867864 1.3752011514738225 -3.5536522205055734 4.203810336447898 -0.6883120456492753) '(193.21298709737695 125.87429502210449 239.08631041475817 321.13647707018345 174.64682477460582) 10 50)
;; (PtolAlea 6)
;nuages (ptolemee '(0.17495987038120603 0.6270317502823318 0.12690590345655736 0.1704305096909972 0.867941858000945 0.07822797462014104) '(2.8642535542403413 -1.5969479765686168 -0.3294785606292612 1.9859590884036553 -2.2237849556352662 -3.8289192483367755) '(352.4059058584383 204.62022625770464 116.99837721325154 7.705846368718606 240.43011657973776 99.5010490333681) 10 50)
;crazy partout (ptolemee '(0.7620648588278608 0.39057996010515833 0.4931291350966862 0.07189957438390623 0.25854163851317324 0.5250534555410099) '(4.4640193310827385 -1.4990298571047003 3.657946528229763 0.029558958540780722 2.7504170271575763 0.24130391933266804) '(334.27282708561575 265.94020893041085 32.71175383163943 219.35879517227187 208.1830105337929 22.35570524326657) 10 50)
;clown blanc (ptolemee '(0.5435942458800864 0.4278294694196982 0.5141896819960133 0.5981086671317157 0.8053458087245856 0.029141891801994092) '(3.8590813965705806 4.863683953627355 2.193195237206659 -2.514373618204843 -2.312653041095758 2.9694553054300235) '(48.77234803284025 131.48386160188142 324.74532225121806 13.43211318074188 124.99966823495863 39.193229450272824) 10 50);; (PtolAlea 7)
;; (PtolAlea 7)
;flamenco (ptolemee '(0.9042130967680608 0.8928239753336161 0.03735396272941299 0.14648299877808058 0.6906421830703359 0.8905453419136728 0.9648037999212813) '(3.662636147183619 1.6209892306867788 2.928489088878946 3.1705931726295375 -0.9982210136916878 -2.9349149538854418 -4.370071011353416) '(169.5647233157693 53.86041539508927 276.9392923208685 133.85762329000536 182.7912156708375 3.9526688381147923 93.0690293257479) 10 50)
;couple (ptolemee '(0.8358049855303112 0.44166507281673156 0.8952233882531959 0.30941846465583606 0.021542564074588127 0.14490163912047005 0.42870782209319586) '(3.0132167190559525 0.8626941514437911 2.580734747612354 0.55342487162617 2.605803623061261 -1.3212723030173636 -3.0013052336895933) '(9.995903689406633 182.53585193778545 196.97053998144443 236.4779902183972 326.6587685917668 290.2260203850009 338.28118126519246) 10 50);; (PtolAlea 11)
;; (PtolAlea 8)
;OVNI de biais (ptolemee '(0.9426193944041047 0.9084301943116453 0.35745926640454545 0.42035224553263234 0.07050488902349936 0.3354040334660635 0.6135867249473682 0.8180131033398456) '(-4.912857897417355 -0.8616859575019884 1.721689676347542 4.780077058412278 3.6279522415771197 -3.451220204031505 2.449691514390877 -4.0833511931511595) '(0.042824149211541096 150.04706544810358 325.69810324478544 214.03781282536278 211.01702587868465 260.2247950066343 93.28077229350949 184.59807370714486) 10 50)
;H (ptolemee '(0.12829675735588514 0.7016738964275903 0.2766546742555772 0.0082435235534674 0.5088153137058304 0.4023546528115024 0.09067872556052865 0.8099069442784543) '(-3.967896977379726 -1.7964075259797414 0.9419843665174854 -2.0046616103586956 -4.803201537212733 0.7307090344479477 3.793797341722618 4.842736813878151) '(200.21877758245685 181.05812844086088 21.2090401001443 359.20901656003485 93.39492979160497 289.64459458299837 354.53265285963687 63.54903180813236) 10 50)
;; (PtolAlea 11)
;Sixties (ptolemee '(0.31939640886969195 0.20994810799127092 0.8438184332918719 0.41040879378171413 0.9770351405759066 0.514676391174414 0.2996348893263096 0.6359817285760897 0.7621398619602245 0.6114171661649219 0.8144683126629672) '(-4.5941710650812295 -1.8456606135622793 4.748348928519244 -4.368127043113349 1.309021741727408 1.3727777011185474 3.841417373793977 1.3521528548601136 2.3175827177568955 -0.8553303514706538 2.6135591931893662) '(136.03041175013485 229.8835433086074 303.74869164777675 7.1324732287460435 34.92845742949929 120.16795367033771 232.48722462446844 324.8842298330445 40.19439682400673 302.2897717771861 224.9572848993489) 10 50)
;L'angle (ptolemee '(0.5691625374726254 0.27213126657927644 0.5386914173861805 0.7942020013820418 0.30518456232768404 0.12020113900341306 0.570299415980092 0.9967150576351491 0.47167077748416136 0.3721157747732769 0.7849426638054525) '(2.7060986826309907 0.8028053414128822 4.806125536199893 1.6921532278734315 -2.0418370273743514 -0.72346172430266 3.6612500411667863 4.714048280779362 -4.333393609772238 2.5757075428510223 -3.611363300329751) '(59.92708068327435 321.95332706338417 326.21125475729116 40.25950543630776 67.62375993211523 277.2228909977207 182.6893782293226 192.98802419190858 195.95809056602064 202.0283252038589 203.25777874003683) 10 50)
;; (PtolAlea 16)
;crazy partout (ptolemee '(0.913140561269858 0.3792067065447694 0.8563165717875894 0.9719211942924582 0.7434450252790843 0.07954742215489488 0.867079347124471 0.5338476515074216 0.31590470730495585 0.8047832321464703 0.8579157085305372 0.35107646202869053 0.1874410047893245 0.9497020150529258 0.2390942335856971 0.2833560244801566) '(3.953601250338556 -0.8867523742592933 1.344354704663818 -3.8653906413640655 2.0143751744448677 -1.604715784275375 -2.3272695901380285 -0.20367042199460883 3.0226775780742567 1.0842138269622135 1.3098633034338487 0.9259024372806826 -3.1108178163148796 3.2023194957567753 -1.6004344385125386 -2.358693140295694) '(173.68508520291203 283.93303499358336 262.55720776050765 128.2023639384701 320.441815779436 175.7917042370731 167.62945385208252 216.50884951745877 196.53971351038612 185.1145209839838 182.11168807393256 207.7482774252971 101.68173324629987 272.9219279306418 187.13315819787448 155.68250193387738) 10 50)
;idem + fin (ptolemee '(0.913140561269858 0.3792067065447694 0.8563165717875894 0.9719211942924582 0.7434450252790843 0.07954742215489488 0.867079347124471 0.5338476515074216 0.31590470730495585 0.8047832321464703 0.8579157085305372 0.35107646202869053 0.1874410047893245 0.9497020150529258 0.2390942335856971 0.2833560244801566) '(3.953601250338556 -0.8867523742592933 1.344354704663818 -3.8653906413640655 2.0143751744448677 -1.604715784275375 -2.3272695901380285 -0.20367042199460883 3.0226775780742567 1.0842138269622135 1.3098633034338487 0.9259024372806826 -3.1108178163148796 3.2023194957567753 -1.6004344385125386 -2.358693140295694) '(173.68508520291203 283.93303499358336 262.55720776050765 128.2023639384701 320.441815779436 175.7917042370731 167.62945385208252 216.50884951745877 196.53971351038612 185.1145209839838 182.11168807393256 207.7482774252971 101.68173324629987 272.9219279306418 187.13315819787448 155.68250193387738) 10 100)
;silhouettes (ptolemee '(0.6675077745085727 0.8319163058547607 0.8293517061124588 0.8026733748664584 0.08246904068155035 0.23521212159695498 0.31578493735899205 0.8432469849348944 0.4968376951233255 0.5376570942581763 0.8678400122892856 0.42043211948207043 0.3021229790563192 0.3583750079541933 0.7042537158434349 0.7589981719087775) '(3.26261616919677 2.456439188600404 -4.780335500427215 2.5471979892760475 1.219031040033384 -1.3241448929983095 -1.6161186557563365 -1.297764212447711 -4.56535369076832 -2.1317136047998697 2.9657032302873265 -0.426251581732652 4.491626898742517 -0.30483118315195945 2.139594791717072 1.3055487467079399) '(294.34144834865157 57.173306973973865 113.29231000187306 79.230033083316 232.7274332060058 231.74517272365313 84.86585245442735 268.08274475500497 234.1040629038663 118.95172945959816 114.9278526658711 60.27341019413138 63.32507271938971 348.7026234505747 231.25624960522399 188.27054557999696) 10 50)
;idem + fin (ptolemee '(0.6675077745085727 0.8319163058547607 0.8293517061124588 0.8026733748664584 0.08246904068155035 0.23521212159695498 0.31578493735899205 0.8432469849348944 0.4968376951233255 0.5376570942581763 0.8678400122892856 0.42043211948207043 0.3021229790563192 0.3583750079541933 0.7042537158434349 0.7589981719087775) '(3.26261616919677 2.456439188600404 -4.780335500427215 2.5471979892760475 1.219031040033384 -1.3241448929983095 -1.6161186557563365 -1.297764212447711 -4.56535369076832 -2.1317136047998697 2.9657032302873265 -0.426251581732652 4.491626898742517 -0.30483118315195945 2.139594791717072 1.3055487467079399) '(294.34144834865157 57.173306973973865 113.29231000187306 79.230033083316 232.7274332060058 231.74517272365313 84.86585245442735 268.08274475500497 234.1040629038663 118.95172945959816 114.9278526658711 60.27341019413138 63.32507271938971 348.7026234505747 231.25624960522399 188.27054557999696) 10 200)
;; Pourquoi pas ?! (PtolAlea 87)















