;;; Canons stochastiques ralisant des squences de termes rpartis
;; quasi-alatoirement

;;   Gestion du module "canons-stochastiques", ncessaire pour la fonction
;; init-random (ainsi que frandom, si utilise).

(when (not (boundp 'common-lisp::*path-stochastique*))
	(setq common-lisp::*path-stochastique* (pathname ""))
	(format *error-output*
		"Attention : le path common-lisp::*path-stochastique* est nul :~
		~%~12Terreur(s) fichier(s) possible(s)."))
(require "canons-stochastiques"
	(merge-pathnames "canons_base.lisp" common-lisp::*path-stochastique*))

;;   Squence quasi-alatoire de van der Corput, cf. [HAYES, Brian, 2011 :
;; "Excursions quasi-alatoires", Pour la science : 410 : 59].
;;   Pour engendrer une squence de van der Corput de longueur <nbr> dans [0,1[,
;; on utilise les nombres entiers dans [0,<nbr>[. Chaque nombre est converti en
;; une liste de chiffres binaires, elle-mme rtrograde puis utilise comme
;; partie fractionnelle d'un nombre binaire dans [0,1[. Par exemple, pour le
;; nombre 6 :
;;    (dec2bin 6) => (1 1 0)
;;    (reverse (dec2bin 6)) => (0 1 1)
;;    Cette liste, utilise comme partie fractionnelle, constitue le nombre
;;    binaire 0.011. Celui-ci est converti en rationnel :
;;    (fracbin (reverse (dec2bin 6))) => 3/8
;;   La squence de van der Corput est structure de manire caractristique.
;; Une permutation alatoire permet au besoin de dtruire cette structure, au
;; moyen de l'option :permut T [NIL par dfaut].
;;   Les termes de la squence sont "bien rpartis" dans [0,1[, et mme
;; "parfaitement" rpartis pour tout <nbr> puissance de 2. Pour d'autres
;; valeurs de <nbr>, la rpartition est moins gale.
;;   Les valeurs rendues sont translates dans [<min>,<max>[.
;;   La squence ne comportera pas de doublons si la rsolution disponible dans
;; [<min>,<max>[ le permet. Des effets d'arrondis produiront des doublons dans
;; les autres cas.
;;   Exemples :
;;    (QuAleaCorput 16 0 1)
;;        => (0 1/2 1/4 3/4 1/8 5/8 3/8 7/8 1/16 9/16
;;            5/16 13/16 3/16 11/16 7/16 15/16)
;;    (QuAleaCorput 17 -1 1)
;;        => (-1 0 -1/2 1/2 -3/4 1/4 -1/4 3/4 -7/8 1/8
;;            -3/8 5/8 -5/8 3/8 -1/8 7/8 -15/16)

(defun QuAleaCorput (nbr min max &key permut &aux rez)
  (let ((ambit (- max min)))
    (dotimes (n nbr (if permut (permuter2a rez) (reverse rez)))
      (push (+ min (* ambit (fracbin (reverse (dec2bin n))))) rez))))

;;   Pour reprsenter une squence quasi-alatoire de van der Corput de longueur
;; <nbr> dans [0,1[ sous forme d'entiers dans [0,M[, o M est la premire
;; puissance de 2 suprieure ou gale  <nbr>. Exemples :
;;    - (QuAleaCorput 8 0 1) => (0 1/2 1/4 3/4 1/8 5/8 3/8 7/8)
;;      (tsX (QuAleaCorput 8 0 1)) => (0 4 2 6 1 5 3 7)
;;      (tsX (QuAleaCorput 8 0 1 :permut t)) => (5 6 3 1 0 4 2 7)
;;    - (QuAleaCorput 13 0 1)
;;          => (0 1/2 1/4 3/4 1/8 5/8 3/8 7/8 1/16 9/16 5/16 13/16 3/16)
;;      (tsX (QuAleaCorput 13 0 1)) => (0 8 4 12 2 10 6 14 1 9 5 13 3)
;;      (tsX (QuAleaCorput 13 0 1 :permut t)) => (9 4 3 14 13 6 5 8 1 2 0 10 12)

(defun tsX (li &aux rez)
  (let ((deno (apply #'max (mapcar #'denominator li))))
    (dolist (frac li (reverse rez))
      (push (* (numerator frac) (/ deno (denominator frac))) rez))))

;;   Squence quasi-alatoire de <nbr> termes galement rpartis dans
;; [<min>,<max>]. Seule la permutation alatoire de ces termes donne au rsultat
;; une allure "alatoire". On peut obtenir la liste des termes galement
;; rpartis sans permutation, au moyen de l'option :permut nil.
;;   La squence ne comportera pas de doublons si la rsolution disponible dans
;; [<min>,<max>[ le permet. Des effets d'arrondis produiront des doublons dans
;; les autres cas.
;;   Exemples :
;;    (QuAleaEq 16 0 1)
;;        => (0 1/2 1/4 3/4 1/8 5/8 3/8 7/8 1/16 9/16
;;            5/16 13/16 3/16 11/16 7/16 15/16)
;;    (QuAleaEq 17 -1 1)
;;        => (-1 0 -1/2 1/2 -3/4 1/4 -1/4 3/4 -7/8 1/8
;;            -3/8 5/8 -5/8 3/8 -1/8 7/8 -15/16)

(defun QuAleaEq (nbr min max &key (permut t) &aux rez)
  (let ((denom (1- nbr))
        (ambit (- max min)))
    (dotimes (n nbr (if permut (permuter2a rez) (reverse rez)))
      (push (+ min (* ambit (/ n denom))) rez))))

;;   Pour reprsenter une squence dans [-1,1] sous forme "graphique". La
;; fonction ligne est dfinie dans ped.10.demo.
;;   Exemples :
;;    (graf (QuAleaCorput 10 -1 1 :permut t))
;;    (graf (QuAleaCorput 10 -1 1))
;;    (graf (QuAleaEq 10 -1 1))
;;    (graf (QuAleaEq 10 -1 1 :permut nil))

(defun graf (li)
  (dolist (le li 'ok) (ligne le)))

;;   Applications sur un registre de hauteurs MIDI :
;-    (melos1.1 'poi :pitch (mapcar #'round (QuAleaCorput 37 24 96))
;-                   :mm '(1/4 . 300))
;-    (melos1.1 'poi :pitch (mapcar #'round (QuAleaCorput 37 24 96 :permut t))
;-                   :mm '(1/4 . 300))
;-    (melos1.1 'poi :pitch (mapcar #'round (QuAleaCorput 41 24 96))
;-                   :mm '(1/4 . 300))
;-    (melos1.1 'poi :pitch (mapcar #'round (QuAleaCorput 41 24 96 :permut t))
;-                   :mm '(1/4 . 300))

;-    (melos1.1 'poi :pitch (mapcar #'round (QuAleaEq 37 24 96))
;-                   :mm '(1/4 . 300))
;-    (melos1.1 'poi :pitch (mapcar #'round (QuAleaEq 41 24 96))
;-                   :mm '(1/4 . 300))

;;; Utilitaires

;;   Conversion d'un nombre dcimal en liste de chiffres binaires. Cf.
;; AutoCell/AutoCell.

(defun dec2bin (nbr &aux (tmp 0))
  (let* ((lng (if (<= nbr 0) 0 (floor (log nbr 2))))
         (rez (make-list (1+ lng) :initial-element 0)))
    (labels ((interne (n)
               (if (<= n 0) (reverse rez)
               (progn (setf (nth (setq tmp (floor (log n 2))) rez) 1)
                      (interne (- n (expt 2 tmp)))))))
      (if (and (integerp nbr) (> nbr 0)) (interne nbr) rez))))

;;   Conversion d'une liste de chiffres fractionnels binaires en rationnel.

(defun fracbin (lichifr &aux (cum 0) (cpt 0))
  (let ((cumu 0)
        (cmpt 0))
    (dolist (chifr lichifr cumu)
      (incf cumu (* chifr (expt 2 (decf cmpt)))))))

;;   Permutation alatoire : identique  permuter2 de ped.16.3a.

(defun permuter2a (liste)
  (let* ((perm ())
         (longueur (length liste))
         (orig (make-array longueur :initial-contents liste))
         (ic (1- longueur))
         (trav (make-array longueur
                           :initial-contents
                              (do* ((indx 0 (1+ indx))
                                    (listindx (list indx) (push indx listindx)))
                                   ((= indx ic) listindx)))))
    (do (ix)
        ((= ic -1) perm)
      (setq ix (random (1+ ic)))
      (push (aref orig (aref trav ix)) perm)
      (setf (aref trav ix) (aref trav ic))
      (decf ic))))

;;   "Documentation".

(setq Doc.Stoch (concatenate 'string Doc.Stoch
                                     (format nil ";; QuAlea (quasi-aleatoires)
;    QuAleaCorput <nbr> <min> <max> [:permut t]
;        => liste de <nbr> termes de seq. de van der Corput dans [<min>,<max>[
;    QuAleaEQ <nbr> <min> <max> [:permut nil]
;        => liste de <nbr> termes galement rpartis dans [<min>,<max>]
")))

;(format t "~A" Doc.Stoch)
