;;   Algorithmes d'alternances dterministes, d'allure "quasi-alatoire" et
;; auto-similaire, bass sur le mot de Thue-Morse.
;;   Cf. ASSOUS, Roland, "Deux ou trois remarques sur l'utilisation des mots
;; de Thue-Morse en informatique musicale", <Musique et Mathmatiques>, Lyon,
;; GRAME, 1996, pp. 80-84.

;;   La programmation objet en CLOS est justifie par la ncessit, dans ces
;; processus dterministes, de dfinir la nouvelle lettre de chaque mot  partir
;; de la mmoire conserve des lettres prcdentes. Il est donc ncessaire que
;; des objets, chacun pelant un mot de Thue-Morse indpendamment, grent chacun
;; sa propre mmorisation des lettres passes.

;;   La fonction gnrique suivante : ncessaire en MCL pour assurer la
;; documentation-string des diffrentes mthodes laisse ci-dessous.

(defgeneric laisse (x) (:documentation
"
;    laisse <nom> => le n-ime appel donnera la n-ime lettre du mot de
;                    Thue-Morse correspondant.
"))

;;   Appels :
;;    cree-thue <nom> <liste> => cration d'un objet thue<N> de nom <nom>,
;;                               initialisation de ses variables d'instance
;;                               <state> et <topop>  (t nil), et de son
;;                               alphabet au contenu de <liste> de longueur
;;                               possible parmi {2,3,4,6,10}
;;    laisse <nom>            => le n-ime appel donnera la n-ime lettre du
;;                               mot de Thue-Morse correspondant

(defun cree-thue (nom liste)
"
;    cree-thue <nom> <liste> => creation d'un objet thue<N> de nom <nom>,
;                               initialisation de ses variables d'instance state
;                               et topop a (t nil), et de son alphabet au
;                               contenu de <liste> de longueur parmi {2,3,4,6,10}.
"
  (let ((nbr (list-length liste))) (cond
       ((=  2 nbr) (eval `(defparameter ,nom (make-instance 'thue2
                              :table (make-array ,nbr
                                       :initial-contents ',liste)))))
       ((=  3 nbr) (eval `(defparameter ,nom (make-instance 'thue3
                              :table (make-array ,nbr
                                       :initial-contents ',liste)))))
       ((=  4 nbr) (eval `(defparameter ,nom (make-instance 'thue4
                              :table (make-array ,nbr
                                       :initial-contents ',liste)))))
       ((=  6 nbr) (eval `(defparameter ,nom (make-instance 'thue6
                              :table (make-array 7
                                       :initial-contents
                                          (list nil
                                                (nth 3 ',liste) ;ordre bizarre :
                                                (nth 1 ',liste) ;pour viter un
                                                (nth 5 ',liste) ;reverse dans la
                                                (nth 0 ',liste) ;mthode laisse.
                                                (nth 4 ',liste)
                                                (nth 2 ',liste)))))))
       ((= 10 nbr) (eval `(defparameter ,nom (make-instance 'thue10
                              :table (make-array 16
                                       :initial-contents
                                          (list nil nil
                                                (nth 2 ',liste) ;ordre bizarre :
                                                (nth 8 ',liste) ;pour viter un
                                                (nth 0 ',liste) ;reverse dans la
                                                (nth 6 ',liste) ;mthode laisse.
                                                (nth 4 ',liste)
                                                nil nil
                                                (nth 5 ',liste)
                                                (nth 3 ',liste)
                                                (nth 9 ',liste)
                                                (nth 1 ',liste)
                                                (nth 7 ',liste)
                                                nil nil))
                              :state '(t nil nil t)
                              :topop '(t nil nil t)))))
       (t          (error "Mots de Thue-Morse seulement pour des alphabets ~
                           de~%2, 3, 4, 6 et 10 lettres.")))))

;;   Seuls les alphabets de longueurs suivantes sont admis : 2, 3, 4, 6 et 10.

;; 2
;;   Mot de Thue-Morse proprement dit. Il s'agit d'un mot compos de deux
;; lettres, infini, sans cube, dans lequel tout facteur a une infinit
;; d'occurrences. On obtient une nouvelle lettre  chaque appel de la mthode
;; laisse. On forme le mme mot si l'on retient toutes les lettres, ou toutes
;; les 2, ou toutes les 4, etc. On forme deux mots complmentaires selon que
;; l'on dbute sur une lettre paire ou impaire. Remise  "zro" du processus par
;; cration d'un nouvel objet thue2. Deux variables d'instance, state et topop,
;; servant de mmoire au processus, sont compltes au fur et  mesure des
;; besoins. Leur taille crot en puissances successives de 2 ; mais la ncessit
;; de complter ces variables est videmment de moins en moins frquente 
;; mesure que leur taille augmente. Appels :
;;    cree-thue <nom> <liste> => cration d'un objet thue2 de nom <nom>, et
;;                               initialisation de son alphabet au contenu de
;;                               la <liste> de longueur 2 spcifiant les
;;                               lettres <l1> et <l2>
;;    laisse <nom>            => le n-ime appel donnera <l1> ou <l2>, n-ime
;;                               lettre du mot de Thue-Morse

(defclass thue2
  ()
  ((state :initarg :state)
   (topop :initarg :topop)
   (table :initarg :table))
  (:default-initargs :state '(t nil)
                     :topop '(t nil)
                     :table (make-array 2)))

(defmethod laisse ((obj thue2))
  (let ((state (slot-value obj 'state))
        (topop (slot-value obj 'topop)))
  (when (endp topop)
    (setq state (append state (setq topop (list-not state))))
    (reinitialize-instance obj :state state)
    (reinitialize-instance obj :topop topop))
  (reinitialize-instance obj :topop (cdr topop))
  (if (pop topop) (aref (slot-value obj 'table) 0)
  (aref (slot-value obj 'table) 1))))

;;   Fonction auxiliaire : cre une liste complmentaire de la liste reue en
;; argument : les t sont changs en nil, et vice-versa.

(defun list-not (li)
  (if (endp li) ()
  (cons (not (car li)) (list-not (cdr li)))))


;; 3
;;   Mot infini sans carr, dduit du mot de Thue-Morse "thue2" ci-dessus,
;; construit sur 3 lettres quiprobables : 1/3 chacune. Appels :
;;    cree-thue <nom> <liste> => cration d'un objet thue3 de nom <nom>, et
;;                               initialisation de son alphabet au contenu de
;;                               la <liste> de longueur 3 spcifiant les
;;                               lettres <l1>, <l2>, <l3>
;;    laisse <nom>            => le n-ime appel donnera <l1>, ou <l2>, ou
;;                               <l3>, n-ime lettre du mot de Thue-Morse

(defclass thue3
  (thue2) nil)

(defmethod laisse ((obj thue3))
  (let ((state (slot-value obj 'state))
        (topop (slot-value obj 'topop))
        x1
        x2)
    (setq x1 (pop topop))
    (when (endp topop)
      (setq state (append state (setq topop (list-not state))))
      (reinitialize-instance obj :state state))
    (setq x2 (car topop))
    (reinitialize-instance obj :topop topop)
    (if x1 (if x2 (aref (slot-value obj 'table) 0)
           (aref (slot-value obj 'table) 1))
    (if x2 (aref (slot-value obj 'table) 2)
    (aref (slot-value obj 'table) 0)))))


;; 4
;;   Mot infini sans carr, dduit du mot de Thue-Morse "thue2" ci-dessus,
;; construit sur 4 lettres. Les lettres 1 et 4 sont moins frquentes : 1/6
;; chacune, contre 1/3 chacune pour les 2 et 3. Appels :
;;    (cree-thue <nom> <liste> => cration d'un objet thue4 de nom <nom>, et
;;                                initialisation de son alphabet au contenu de
;;                                la <liste> de longueur 4 spcifiant les
;;                                lettres <l1>, <l2>, <l3> et <l4>
;;    laisse <nom>             => le n-ime appel donnera <l1>, ou <l2>, etc.,
;;                                n-ime lettre du mot de Thue-Morse

(defclass thue4
  (thue2) nil)

(defmethod laisse ((obj thue4))
  (let ((state (slot-value obj 'state))
        (topop (slot-value obj 'topop))
        x1
        x2)
    (setq x1 (pop topop))
    (when (endp topop)
      (setq state (append state (setq topop (list-not state))))
      (reinitialize-instance obj :state state))
    (setq x2 (car topop))
    (reinitialize-instance obj :topop topop)
    (if x1 (if x2 (aref (slot-value obj 'table) 0)
           (aref (slot-value obj 'table) 1))
    (if x2 (aref (slot-value obj 'table) 2)
    (aref (slot-value obj 'table) 3)))))

;; 6 SANS CARR ?????
;;   Mot infini sans carr, dduit du mot de Thue-Morse "thue2" ci-dessus,
;; construit sur 6 lettres quiprobables : 1/6 chacune.
;;   Squence type :
;;    36524125364136524136412536413652412536524136412536413652413641253652412536413652413641253641365241253652413641253652412536413652412536524136412536413652413641253652412536413652413641253641365241253652413641253641365241364125365241253641365241253652413641253652412536413652413641253641365241253652413641253641365241364125365241253641365241364125364136524125365241364125365241253641365241253652413641253641365241364125365241253641365241253652413641253652412536413652413641253641365241253652413641253652412536413652412536524136412536413652413641253652412536413652413641253641365241253652413641253641365241364125365241253641365241253652413641253652412536413652413641253641365241253652413641253652412536413652412536524136412536413652413641253652412536413652412536524136412536524125364136524136412536413652412536524136412536413652413641253652412536413652413641253641365241253652413641253652412536413652412536524136412536413652413641253652412536413652412536524136412536524125364136524136412536413652412536524136412536524125364136524125365241364125364136524136412536524125364136524136412536413652412536524136412536413652413641253652412536413652412536524136412536524125364136524136412536413652412536524136412536413652413641253652412536413652413641253641365241253652413641253652412536413652412536524136412536413652413641253652412536413652413641253641365241253652413641253641365241364125365241253641365241253652413641253652412536413652413641253641365241253652413641253652412536413652412536524136412536413652413641253652412536413652412536524136412536524125364136524136412536413652412536524136412536413652413641253652412536413652413641253641365241253652413641253652412536413652412536524136412536413652413641253652412536413652413641253641365241253652413641253641365241364125365241253641365241253652413641253652412536413652413641253
;;   Appels :
;;    cree-thue <nom> <liste> => cration d'un objet thue6 de nom <nom>, et
;;                               initialisation de son alphabet au contenu de
;;                               la <liste> de longueur 6 spcifiant les
;;                               lettres <l1>, <l2>, <l3>, <l4>, <l5> et <l6>
;;     laisse <nom>           => le n-ime appel donnera <l1>, ou <l2>, etc.,
;;                               n-ime lettre du mot de Thue-Morse

(defclass thue6
  (thue2) nil)

(defmethod laisse ((obj thue6))
  (let ((state (slot-value obj 'state))
        (topop (slot-value obj 'topop))
        x1)
    (push (pop topop) x1)
    (push (car topop) x1)
    (when (endp (cdr topop))
      (setq state (append state (cdr (setq topop (append topop
                                                         (list-not state))))))
      (reinitialize-instance obj :state state))
    (push (cadr topop) x1)
    (reinitialize-instance obj :topop topop)
    (aref (slot-value obj 'table) (bin>dec x1))))

;;   Fonction auxiliaire : convertit une liste de t et nil, reprsentant un
;; nombre binaire (o t=1 et nil=0) en valeur dcimale (cf. [Wertz:50]).

(defun bin>dec (li)
  (labels ((auXil (li res)
             (if li (auXil (cdr li) (+ (* res 2) (if (car li) 1 0)))
             res)))
    (auXil li 0)))


;;   Ces trois lignes pour aller avec l'essai manuscrit du train Valognes.

(cree-thue 'six '(1 2 3 4 5 6))
(reinitialize-instance six :topop '(nil t))
(reinitialize-instance six :state '(nil t))

;; 10 SANS CARR ?????
;;   Mot infini sans carr, dduit du mot de Thue-Morse "thue2" ci-dessus,
;; construit sur 10 lettres de probabilits ingales : les lettres 5 et 6 ont
;; chacune une probabilit 1/6, tandis que les autres (1  4 d'une part, et 7
;;  10 d'autre part) ont 1/12 chacune.
;;   Squence type :
;;    3614850736259614859625073625961485073614859625073614850736259614850736148596250736259614859625073614850736259614859625073625961485073614859625073625961485962507361485073625961485073614859625073614850736259614859625073625961485073614859625073625961485962507361485073625961485962507362596148507361485962507361485073625961485073614859625073625961485962507361485073625961485962507362596148507361485962507362596148596250736148507362596148507361485962507361485073625961485962507362596148507361485962507361485073625961485073614859625073625961485962507361485073625961485073614859625073614850736259614859625073625961485073614859625073625961485962507361485073625961485962507362596148507361485962507361485073625961485073614859625073625961485962507361485073625961485073614859625073614850736259614859625073625961485073614859625073614850736259614850736148596250736259614859625073614850736259614859625073625961485073614859625073625961485962507361485073625961485073614859625073614850736259614859625073625961485073614859625073614850736259614850736148596250736259614859625073614850736259614850736148596250736148507362596148596250736259614850736148596250736259614859625073614850736259614859625073625961485073614859625073614850736259614850736148596250736259614859625073614850736259614
;;   Appels :
;;    cree-thue <nom> <liste> => cration d'un objet thue10 de nom <nom>, et
;;                               initialisation de son alphabet au contenu de
;;                               la <liste> de longueur 10 spcifiant les
;;                               lettres <l1>, <l2>, ... <l10>
;;    laisse <nom>            => le n-ime appel donnera <l1>, ou <l2>, etc.,
;;                               n-ime lettre du mot de Thue-Morse

(defclass thue10
  (thue2) nil)

(defmethod laisse ((obj thue10))
  (let ((state (slot-value obj 'state))
        (topop (slot-value obj 'topop))
        x1)
    (push (pop topop) x1)
    (push (car topop) x1)
    (push (cadr topop) x1)
    (when (endp (cddr topop))
      (setq state (append state (cddr (setq topop (append topop
                                                          (list-not state))))))
      (reinitialize-instance obj :state state))
    (push (caddr topop) x1)
    (reinitialize-instance obj :topop topop)
    (aref (slot-value obj 'table) (bin>dec x1))))


;;   Ces trois lignes pour aller avec l'essai manuscrit du train Valognes.
(cree-thue 'dix '(2 3 4 5 6 9 10 11 12 13))
(reinitialize-instance dix :topop '(nil t t nil))
(reinitialize-instance dix :state '(nil t t nil))


;;   Pour vrifier les rpartitions de probabilits... Il faut que les lettres
;; de l'alphabet soient utilisables comme indices d'un vecteur, c'est--dire des
;; nombres allant de 0  <N>-1 pour un alphabet de <N> lettres.
;;   Exemple d'appel, si l'alphabet de l'objet dix est '(0 1 2 3 4 5 6 7 8 9) :
;;    (statist 10 '(laisse dix) 1000)
;; Cette fonction est plus gnralement utile pour vrifier des fonctions de
;; probabilits quelconques. Par exemple :
;;    (statist 10 '(floor (random 1.0) 0.1) 10000)

(defun statist (choses forme fois)
  (let ((vect (make-array choses :initial-element 0))
        tmp)
    (dotimes (i fois)
      (setq tmp (eval forme))
      (setf (aref vect tmp) (1+ (aref vect tmp))))
    vect))

;;   "Documentation".

(setq Doc.Stoch (concatenate 'string Doc.Stoch
                                     (format nil ";; thue-morse.clos
;    cree-thue <nom> <liste>   => objet thue<N> nomme <nom> ; init. de ses vars.
;                                 d'instance state et topop := (t nil) ; son
;                                 alphabet := <liste> de long. parmi {2,3,4,6,10}
;    laisse <nom>              => n-ieme lettre du mot de Thue-Morse <nom>
")))

;(format t "~A" Doc.Stoch)
